/* SPDX-License-Identifier: GPL-2.0-only */

#include <security/vboot/antirollback.h>
#include <program_loading.h>
#include <vb2_api.h>
#include <security/tpm/tss.h>
#include <security/vboot/misc.h>
#include <security/vboot/mrc_cache_hash_tpm.h>
#include <console/console.h>
#include <string.h>

void mrc_cache_update_hash(uint32_t index, const uint8_t *data, size_t size)
{
	struct vb2_hash hash;
	tpm_result_t rc = TPM_SUCCESS;

	/* Initialize TPM driver. */
	rc = tlcl_lib_init();
	if (rc != TPM_SUCCESS) {
		printk(BIOS_ERR, "MRC: TPM driver initialization failed with error %#x.\n", rc);
		return;
	}

	/* Calculate hash of data generated by MRC. */
	if (vb2_hash_calculate(vboot_hwcrypto_allowed(), data, size,
			       VB2_HASH_SHA256, &hash)) {
		printk(BIOS_ERR, "MRC: SHA-256 calculation failed for data. "
		       "Not updating TPM hash space.\n");
		/*
		 * Since data is being updated in mrc cache, the hash
		 * currently stored in TPM hash space is no longer
		 * valid. If we are not able to calculate hash of the
		 * data being updated, reset all the bits in TPM hash
		 * space to zero to invalidate it.
		 */
		memset(hash.raw, 0, VB2_SHA256_DIGEST_SIZE);
	}

	/* Write hash of data to TPM space. */
	rc = antirollback_write_space_mrc_hash(index, hash.sha256, sizeof(hash.sha256));
	if (rc != TPM_SUCCESS) {
		printk(BIOS_ERR, "MRC: Could not save hash to TPM with error %#x.\n", rc);
		return;
	}

	printk(BIOS_INFO, "MRC: TPM MRC hash idx %#x updated successfully.\n", index);
}

int mrc_cache_verify_hash(uint32_t index, const uint8_t *data, size_t size)
{
	struct vb2_hash tpm_hash = { .algo = VB2_HASH_SHA256 };
	tpm_result_t rc = TPM_SUCCESS;

	/* Initialize TPM driver. */
	rc = tlcl_lib_init();
	if (rc != TPM_SUCCESS) {
		printk(BIOS_ERR, "MRC: TPM driver initialization failed with error %#x.\n", rc);
		return 0;
	}

	/* Read hash of MRC data saved in TPM. */
	rc = antirollback_read_space_mrc_hash(index, tpm_hash.sha256, sizeof(tpm_hash.sha256));
	if (rc != TPM_SUCCESS) {
		printk(BIOS_ERR, "MRC: Could not read hash from TPM with error %#x.\n", rc);
		return 0;
	}

	/* Calculate hash of data read from MRC_CACHE and compare. */
	if (vb2_hash_verify(vboot_hwcrypto_allowed(), data, size, &tpm_hash)) {
		printk(BIOS_ERR, "MRC: Hash comparison failed.\n");
		return 0;
	}

	printk(BIOS_INFO, "MRC: Hash idx %#x comparison successful.\n", index);

	return 1;
}
